#requires -Module ActiveDirectory

[CmdletBinding(ConfirmImpact = 'High', SupportsShouldProcess)]
param (
	[Parameter(Mandatory)]
	[ValidateNotNullOrEmpty()]
	[string]$FirstName,

	[Parameter(Mandatory)]
	[ValidateNotNullOrEmpty()]
	[string]$LastName,

	[Parameter(Mandatory)]
	[ValidateNotNullOrEmpty()]
	[string]$Department,

	[Parameter(Mandatory)]
	[ValidateNotNullOrEmpty()]
	[int]$EmployeeNumber
)

try {
	## Pierwsza próba utworzenia konta użytkownika o nazwie składającej się z pierwszej litery imienia i całego nazwiska
	$userName = '{0}{1}' -f $FirstName.Substring(0, 1), $LastName
	
	## Użyj pętli while, aby sprawdzić różne odmiany nazwy użytkownika, jeśli oryginał jest już zajęty. 
	## W takim przypadku, użyj drugiej litery imienia + nazwisko, potem trzeciej litery itd., 
	## aż do momentu, kiedy zostanie znaleziona unikalna nazwa konta użytkownika.
	$i = 2
	while ((Get-AdUser -Filter "samAccountName -eq '$userName'") -and ($userName -notlike "$FirstName*")) {
		Write-Warning -Message "The username [$($userName)] already exists. Trying another..."
		$userName = '{0}{1}' -f $FirstName.Substring(0, $i), $LastName
		Start-Sleep -Seconds 1
		$i++
	}

	## Upewnij się, że nie wyczerpaliśmy wszystkich możliwości nazwy konta użytkownika
	if ($userName -like "$FirstName*") {
		throw 'Nie udało się znaleźć unikalnej nazwy konta użytkownika'
	} elseif (-not ($ou = Get-ADOrganizationalUnit -Filter "Name -eq '$Department'")) {
		## Sprawdź, czy istnieje OU odpowiadające nazwie działu
		throw "Active Directory OU dla działu [$($Department)] nie zostało znalezione."
	} elseif (-not (Get-AdGroup -Filter "Name -eq '$Department'")) {
		throw "Grupa [$($Department)] nie istnieje."
	} else {
		## Utwórz losowe hasło
		Add-Type -AssemblyName 'System.Web'
		$password = [System.Web.Security.Membership]::GeneratePassword((Get-Random -Minimum 20 -Maximum 32), 3)
		$secPw = ConvertTo-SecureString -String $password -AsPlainText -Force

		$newUserParams = @{
			GivenName             = $FirstName
			EmployeeNumber        = $EmployeeNumber
			Surname               = $LastName
			Name                  = $userName
			AccountPassword       = $secPw
			ChangePasswordAtLogon = $true
			Enabled               = $true
			Department            = $Department
			Path                  = $ou.DistinguishedName
			Confirm               = $false
		}
		if ($PSCmdlet.ShouldProcess("AD user [$userName]", "Create AD user $FirstName $LastName")) {
			## Utwórz konto użytkownika
			New-AdUser @newUserParams
			
			## Dodaj użytkownika do grupy reprezentującej jego dział
			Add-AdGroupMember -Identity $Department -Members $userName

			[pscustomobject]@{
				FirstName      = $FirstName
				LastName       = $LastName
				EmployeeNumber = $EmployeeNumber
				Department     = $Department
				Password       = [System.Runtime.InteropServices.marshal]::PtrToStringAuto([System.Runtime.InteropServices.marshal]::SecureStringToBSTR($secPw))
			}
		}
	}
} catch {
	Write-Error -Message $_.Exception.Message
}